<?php
/**
 * This is a PHP library that handles calling reCAPTCHA.
 *
 * @ignore
 * @copyright Copyright (c) 2015, Google Inc.
 * @link      http://www.google.com/recaptcha
 * @internal
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 *
 * @codingStandardsIgnoreFile
 */

namespace ReCaptcha\RequestMethod;

use ReCaptcha\RequestMethod;
use ReCaptcha\RequestParameters;

/**
 * Sends POST requests to the reCAPTCHA service.
 *
 * @ignore
 */
class Post implements RequestMethod {

	/**
	 * URL to which requests are POSTed.
	 *
	 * @const string
	 */
	const SITE_VERIFY_URL = 'https://www.google.com/recaptcha/api/siteverify';

	/**
	 * Submit the POST request with the specified parameters.
	 *
	 * @param RequestParameters $params Request parameters
	 * @return string Body of the reCAPTCHA response
	 */
	public function submit( RequestParameters $params ) {
		/**
		 * PHP 5.6.0 changed the way you specify the peer name for SSL context options.
		 * Using "CN_name" will still work, but it will raise deprecated errors.
		 */
		$peer_key = version_compare( PHP_VERSION, '5.6.0', '<' ) ? 'CN_name' : 'peer_name';
		$options  = array(
			'http' => array(
				'header'      => "Content-type: application/x-www-form-urlencoded\r\n",
				'method'      => 'POST',
				'content'     => $params->toQueryString(),
				// Force the peer to validate (not needed in 5.6.0+, but still works)
				'verify_peer' => true,
				// Force the peer validation to use www.google.com
				$peer_key     => 'www.google.com',
			),
		);
		$context = stream_context_create( $options );
		return file_get_contents( self::SITE_VERIFY_URL, false, $context );
	}
}
